<?php
/**
 * Purchase Total
 *
 * @package     AutomatorWP\Integrations\Easy_Digital_Downloads\Triggers\Purchase_Total
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Easy_Digital_Downloads_Purchase_Total extends AutomatorWP_Integration_Trigger {

    public $integration = 'easy_digital_downloads';
    public $trigger = 'easy_digital_downloads_purchase_total';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'User completes a purchase with a total greater than, less than or equal to a specific amount', 'automatorwp-pro' ),
            'select_option'     => __( 'User completes a purchase with a total <strong>greater than, less than or equal</strong> to a specific amount', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. %2$s: Amount. %3$s: Number of times. */
            'edit_label'        => sprintf( __( 'User completes a purchase with a total %1$s %2$s %3$s time(s)', 'automatorwp-pro' ), '{condition}', '{amount}', '{times}' ),
            /* translators: %1$s: Condition. %2$s: Amount. */
            'log_label'         => sprintf( __( 'User completes a purchase with a total %1$s %2$s', 'automatorwp-pro' ), '{condition}', '{amount}' ),
            'action'            => 'edd_after_payment_actions',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'condition' => automatorwp_utilities_condition_option(),
                'amount' => array(
                    'from' => 'amount',
                    'fields' => array(
                        'amount' => array(
                            'name' => __( 'Amount:', 'automatorwp-pro' ),
                            'desc' => __( 'The payment total required. Support decimals.', 'automatorwp-pro' ),
                            'type' => 'text',
                            'attributes' => array(
                                'type' => 'number',
                                'min' => '0',
                            ),
                            'default' => 0
                        )
                    )
                ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param int $payment_id The payment ID
     */
    public function listener( $payment_id ) {

        $payment = edd_get_payment( $payment_id );

        // Bail if payment object not exists
        if( ! $payment ) {
            return;
        }

        $payment_total = $payment->total;
        $user_id = $payment->user_id;

        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'payment_id'    => $payment_id,
            'payment_total' => $payment_total,
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if payment total is not received
        if( ! isset( $event['payment_total'] ) ) {
            return false;
        }

        $total = $event['payment_total'];
        $required_total =  (float) $trigger_options['amount'];

        // Don't deserve if payment total doesn't match with the trigger option
        if( ! automatorwp_number_condition_matches( $total, $required_total, $trigger_options['condition'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

}

new AutomatorWP_Easy_Digital_Downloads_Purchase_Total();